package reflection;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.util.Arrays;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Examples for package {@link java.lang.invoke}.
 *
 * @author Georgiy Korneev (kgeorgiy@kgeorgiy.info)
 */
@SuppressWarnings("unused")
public enum InvokeExample {
    ;

    private static String impl(final String name, final int a, final int b) {
        System.out.format("\t%s called: a=%d, b=%d%n", name, a, b);
        return "hello-" + (a + b);
    }

    public static String publicStaticMethod(final int a, final int b) {
        return impl("publicStaticMethod", a, b);
    }

    @SuppressWarnings("ProtectedMemberInFinalClass")
    protected static String protectedStaticMethod(final int a, final int b) {
        return impl("protectedStaticMethod", a, b);
    }

    /* package-private */ static String packagePrivateStaticMethod(final int a, final int b) {
        return impl("packagePrivateStaticMethod", a, b);
    }

    private static String privateStaticMethod(final int a, final int b) {
        return impl("privateStaticMethod", a, b);
    }

    public static void main(final String[] args) {
        lookupTypes();
        lookupDynamic();
        loop();
    }

    private static void loop() {
        System.out.println("=== Loop");
        final MethodHandle loop = MethodHandles.whileLoop(
                // :NOTE: "Staring from 5"
                MethodHandles.constant(long.class, 5),
                // LoopExample::pred
                LoopExample.find("pred", boolean.class, long.class, int.class, int.class),
                // LoopExample::step
                LoopExample.find("step", long.class, long.class, int.class, int.class)
        );
        call(loop, "LoopExample", 10, 23);
        call(loop, "LoopExample", 3, 23);
    }

    // int i = init();
    // while (i < limit) {
    //      i += step;
    // }
    // return i
    @SuppressWarnings("unused")
    private static final class LoopExample {
        public static boolean pred(final long i, final int step, final int limit) {
            return i < limit;
        }

        public static long step(final long i, final int step, final int limit) {
            return i + step;
        }

        private static MethodHandle find(final String name, final Class<?> rtype, final Class<?>... ptypes) {
            return findMethod(
                    "LoopExample." + name, () -> MethodHandles.lookup().findStatic(
                            LoopExample.class,
                            name, MethodType.methodType(rtype, ptypes)
                    )
            ).orElseThrow();
        }
    }

    private static void lookupTypes() {
        System.out.println("=== Lookup types");
        callMethods("MethodHandles.publicLookup()", MethodHandles.publicLookup());
        callMethods("MethodHandles.lookup()", MethodHandles.lookup());
    }

    private static void callMethods(final String name, final MethodHandles.Lookup lookup) {
        System.out.println(name);
        final Class<?>[] ptypes = {int.class, int.class};
        int i = 0;
        findAndCallStatic("publicStaticMethod", lookup, ptypes, new Object[]{i += 10, i += 10});
        findAndCallStatic("protectedStaticMethod", lookup, ptypes, new Object[]{i += 10, i += 10});
        findAndCallStatic("packagePrivateStaticMethod", lookup, ptypes, new Object[]{i += 10, i += 10});
        findAndCallStatic("privateStaticMethod", lookup, ptypes, new Object[]{i += 10, i += 10});
    }

    private static void findAndCallStatic(
            final String methodName,
            final MethodHandles.Lookup lookup,
            final Class<?>[] ptypes,
            final Object[] args
    ) {
        final String methodDesc = "%s.%s".formatted(InvokeExample.class.getSimpleName(), methodName);
        findAndCall(methodDesc, ptypes, args, () -> lookup.findStatic(
                InvokeExample.class,
                methodName,
                MethodType.methodType(String.class, ptypes)
        ));
    }

    private static void findAndCall(
            final String methodDesc,
            final Class<?>[] ptypes,
            final Object[] args,
            final Finder finder
    ) {
        final String methodType = methodTypeDesc(methodDesc, ptypes);
        findMethod(methodType, finder)
                .ifPresent(method -> call(method, methodDesc, args));
    }

    private static Optional<MethodHandle> findMethod(final String methodType, final Finder finder) {
        try {
            return Optional.of(finder.find());
        } catch (final NoSuchMethodException e) {
            System.out.format("\tMethod %s not found: %s%n", methodType, e.getMessage());
            return Optional.empty();
        } catch (final IllegalAccessException e) {
            System.out.format("\tCannot access %s(): %s%n", methodType, e.getMessage());
            return Optional.empty();
        }
    }

    private static String methodTypeDesc(final String methodDesc, final Class<?>[] ptypes) {
        return "%s(%s)".formatted(
                methodDesc,
                Arrays.stream(ptypes)
                        .map(Class::getSimpleName)
                        .collect(Collectors.joining(", "))
        );
    }

    private static void call(final MethodHandle method, final String methodDesc, final Object... args) {
        final String callDesc = "%s(%s)".formatted(methodDesc, Arrays.deepToString(args));
        final Object result;
        try {
            result = method.invokeWithArguments(args);
        } catch (final Throwable e) {
            System.out.format("\tMethod %s thrown exception: %s (%s)%n", callDesc, e.getClass().getName(), e.getMessage());
            return;
        }
        System.out.format("\tMethod %s returned: %s%n", callDesc, result);
    }


    private static void lookupDynamic() {
        System.out.println("Dynamic methods");
        findConstructor();
        findVirtual();
    }

    private static void findVirtual() {
        System.out.println("--- findVirtual");
        subSequence("hello");
        subSequence(new StringBuilder("hello"));
        subSequence(20);
    }

    private static void subSequence(final Object target) {
        final Class<?>[] ptypes = {int.class, int.class};
        findAndCall(
                "CharSequence.subSequence",
                ptypes,
                new Object[]{target, 1, 3},
                () -> MethodHandles.lookup()
                        .findVirtual(CharSequence.class, "subSequence", MethodType.methodType(CharSequence.class, ptypes))
        );
    }

    private static void findConstructor() {
        System.out.println("--- findConstructor");
        final Class<?>[] ptypes = {char[].class, int.class, int.class};
        findAndCall(
                "String.String",
                ptypes,
                new Object[]{"hello".toCharArray(), 1, 3},
                () -> MethodHandles.lookup()
                        .findConstructor(String.class, MethodType.methodType(void.class, ptypes))
        );
    }

    private interface Finder {
        MethodHandle find() throws NoSuchMethodException, IllegalAccessException;
    }
}
