package lambda;

import java.util.*;
import java.util.function.Consumer;


/**
 * @author Georgiy Korneev (kgeorgiy@kgeorgiy.info)
 */
public class ArrayMap<K, V> extends AbstractMap<K, V> {
    private Object[] entries = new Object[8];
    private int size;
    private int modCount;

    @SuppressWarnings("unchecked")
    private Entry<K, V> entry(final int i) {
        return Map.entry((K) entries[i], (V) entries[i + 1]);
    }

    @Override
    public V put(final K key, final V value) {
        ensureCapacity(size + 1);
        final int i = size * 2;
        entries[i] = key;
        entries[i + 1] = value;
        size++;
        modCount++;
        return null;
    }

    @Override
    public Set<Entry<K, V>> entrySet() {
        return new AbstractSet<>() {
            @Override
            public int size() {
                return size;
            }

            @Override
            public boolean add(final Entry<K, V> entry) {
                put(entry.getKey(), entry.getValue());
                return true;
            }

            @Override
            public Iterator<Entry<K, V>> iterator() {
                return new Iter();
            }

            @Override
            public Spliterator<Entry<K, V>> spliterator() {
                return new Spliter(0, size * 2);
            }
        };
    }

    private void ensureCapacity(final int size) {
        if (entries.length < 2 * size) {
            entries = Arrays.copyOf(entries, 4 * size);
        }
    }

    private class Iter implements Iterator<Entry<K, V>> {
        int expectedModCount = modCount;
        int i;
        int bound = size * 2;
        boolean canRemove;

        @Override
        public boolean hasNext() {
            return i < bound;
        }

        private void checkForComodification() {
            if (expectedModCount != modCount) {
                throw new ConcurrentModificationException();
            }
        }

        @Override
        public Entry<K, V> next() {
            checkForComodification();
            canRemove = true;
            return entry(i += 2);
        }

        @Override
        public void remove() {
            checkForComodification();
            if (!canRemove) {
                throw new IllegalStateException();
            }
            canRemove = false;

            bound -= 2;
            size--;
            entries[i - 2] = entries[bound - 2];
            entries[i - 1] = entries[bound - 1];
            entries[bound - 2] = entries[bound - 1] == null;
            modCount++;
            expectedModCount++;
        }
    }

    private class Spliter implements Spliterator<Entry<K, V>> {
        private final int expectedModCount = modCount;
        private int i;
        private final int bound;

        public Spliter(final int i, final int bound) {
            this.i = i;
            this.bound = bound;
        }

        private void checkForComodification() {
            if (expectedModCount != modCount) {
                throw new ConcurrentModificationException();
            }
        }

        @Override
        public boolean tryAdvance(final Consumer<? super Entry<K, V>> action) {
            checkForComodification();
            if (i < bound) {
                action.accept(entry(i += 2));
                return true;
            }
            return false;
        }

        @Override
        public void forEachRemaining(final Consumer<? super Entry<K, V>> action) {
            checkForComodification();
            for (; i < bound; i += 2) {
                action.accept(entry(i));
            }
        }

        @Override
        public Spliterator<Entry<K, V>> trySplit() {
            checkForComodification();
            final int left = i;
            final int mid = (left + bound) / 4 * 2;
            if (i < mid) {
                i = mid;
                return new Spliter(left, mid);
            }
            return null;
        }

        @Override
        public long getExactSizeIfKnown() {
            return (bound - i) / 2;
        }

        @Override
        public long estimateSize() {
            return getExactSizeIfKnown();
        }

        @Override
        public int characteristics() {
            return SIZED | SUBSIZED | NONNULL;
        }
    }
}
